-- CRM Database Setup
-- Users (clients)
CREATE TABLE IF NOT EXISTS users (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(200) NOT NULL,
    phone VARCHAR(20) NOT NULL,
    email VARCHAR(200) UNIQUE NOT NULL,
    whatsapp VARCHAR(20),
    telegram_id VARCHAR(100),
    password VARCHAR(255) NOT NULL,
    status ENUM('pending','approved','disapproved') DEFAULT 'pending',
    two_fa_enabled TINYINT(1) DEFAULT 0,
    two_fa_code VARCHAR(10),
    two_fa_expires DATETIME,
    reset_token VARCHAR(100),
    reset_expires DATETIME,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

-- Admin users
CREATE TABLE IF NOT EXISTS admins (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(200) NOT NULL,
    email VARCHAR(200) UNIQUE NOT NULL,
    password VARCHAR(255) NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Service categories
CREATE TABLE IF NOT EXISTS service_categories (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(200) NOT NULL,
    slug VARCHAR(200) NOT NULL,
    description TEXT,
    is_active TINYINT(1) DEFAULT 1,
    sort_order INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Services
CREATE TABLE IF NOT EXISTS services (
    id INT AUTO_INCREMENT PRIMARY KEY,
    category_id INT NOT NULL,
    name VARCHAR(200) NOT NULL,
    description TEXT,
    base_price DECIMAL(10,2) NOT NULL,
    postpay_addon DECIMAL(10,2) DEFAULT 100.00,
    has_payment_plans TINYINT(1) DEFAULT 1,
    classes_count INT DEFAULT NULL,
    price_per_class DECIMAL(10,2) DEFAULT NULL,
    is_training TINYINT(1) DEFAULT 0,
    is_active TINYINT(1) DEFAULT 1,
    sort_order INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (category_id) REFERENCES service_categories(id) ON DELETE CASCADE
);

-- Payment plan terms
CREATE TABLE IF NOT EXISTS payment_plans (
    id INT AUTO_INCREMENT PRIMARY KEY,
    plan_type ENUM('prepay','postpay') NOT NULL,
    service_category ENUM('campaign','training','all') DEFAULT 'all',
    terms TEXT NOT NULL,
    is_active TINYINT(1) DEFAULT 1
);

-- Discount options
CREATE TABLE IF NOT EXISTS discount_options (
    id INT AUTO_INCREMENT PRIMARY KEY,
    amount DECIMAL(10,2) NOT NULL,
    label VARCHAR(100),
    is_active TINYINT(1) DEFAULT 1
);

-- Orders
CREATE TABLE IF NOT EXISTS orders (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    service_id INT NOT NULL,
    payment_plan ENUM('prepay','postpay') DEFAULT NULL,
    num_classes INT DEFAULT NULL,
    discount_amount DECIMAL(10,2) DEFAULT 0,
    subtotal DECIMAL(10,2) NOT NULL,
    total_amount DECIMAL(10,2) NOT NULL,
    payment_method ENUM('usdt','upi') NOT NULL,
    payment_screenshot VARCHAR(500),
    status ENUM('pending','confirmed','rejected','completed') DEFAULT 'pending',
    admin_notes TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id),
    FOREIGN KEY (service_id) REFERENCES services(id)
);

-- Selected dates for training
CREATE TABLE IF NOT EXISTS training_dates (
    id INT AUTO_INCREMENT PRIMARY KEY,
    order_id INT NOT NULL,
    class_date DATE NOT NULL,
    class_time TIME,
    FOREIGN KEY (order_id) REFERENCES orders(id) ON DELETE CASCADE
);

-- Email logs
CREATE TABLE IF NOT EXISTS email_logs (
    id INT AUTO_INCREMENT PRIMARY KEY,
    to_email VARCHAR(200),
    subject VARCHAR(500),
    body TEXT,
    sent_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Settings
CREATE TABLE IF NOT EXISTS settings (
    id INT AUTO_INCREMENT PRIMARY KEY,
    setting_key VARCHAR(100) UNIQUE NOT NULL,
    setting_value TEXT,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

-- Insert default admin
INSERT INTO admins (name, email, password) VALUES 
('Admin', 'admin@crm.com', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi')
ON DUPLICATE KEY UPDATE id=id;
-- default password: password

-- Insert service categories
INSERT INTO service_categories (name, slug, description, sort_order) VALUES
('Campaign Setup', 'campaign', 'Professional ad campaign setup and management', 1),
('Training', 'training', 'Expert-led training classes for digital marketing', 2)
ON DUPLICATE KEY UPDATE id=id;

-- Insert campaign services
INSERT INTO services (category_id, name, base_price, postpay_addon, has_payment_plans, is_training, sort_order) VALUES
(1, 'Google Ads Campaign (Pop Up & PPC)', 399.00, 100.00, 1, 0, 1),
(1, 'Meta Ads Campaign (Pop Up & PPC)', 399.00, 100.00, 1, 0, 2),
(1, 'Combo Campaign (Pop Up & PPC)', 599.00, 100.00, 1, 0, 3),
(1, 'Other Networks (PPC, Push & SMS)', 249.00, 100.00, 1, 0, 4)
ON DUPLICATE KEY UPDATE id=id;

-- Insert training services
INSERT INTO services (category_id, name, base_price, classes_count, price_per_class, has_payment_plans, is_training, sort_order) VALUES
(2, 'Smart POPUP', 69.00, 7, 69.00, 0, 1, 1),
(2, 'Smart PPC', 69.00, 7, 69.00, 0, 1, 2),
(2, 'Experts Pro POPUP', 95.00, 14, 95.00, 0, 1, 3),
(2, 'Experts Pro PPC', 95.00, 14, 95.00, 0, 1, 4),
(2, 'Push (iOS) & SMS', 85.00, 6, 85.00, 0, 1, 5)
ON DUPLICATE KEY UPDATE id=id;

-- Insert payment plan terms
INSERT INTO payment_plans (plan_type, service_category, terms) VALUES
('prepay', 'campaign', '100% payment required in advance along with Pre Requisite Tools before campaign setup begins.'),
('postpay', 'campaign', '25% Advance on Post Pay + Ad A/C Fees + 3 Days Ad Recharge Fees of "Min. $100/Day" + Admin access of an unused or fresh TFN & 75% After 3 Days of Leads/Calls from Ad Campaign.')
ON DUPLICATE KEY UPDATE id=id;

-- Insert discount options
INSERT INTO discount_options (amount, label) VALUES
(10.00, '$10 Discount'),
(20.00, '$20 Discount'),
(30.00, '$30 Discount')
ON DUPLICATE KEY UPDATE id=id;

-- Insert default settings
INSERT INTO settings (setting_key, setting_value) VALUES
('site_name', 'ProCRM'),
('site_email', 'info@procrm.com'),
('usdt_address', 'YOUR_USDT_ADDRESS_HERE'),
('upi_id', 'yourname@upi'),
('smtp_host', 'smtp.gmail.com'),
('smtp_port', '587'),
('smtp_user', ''),
('smtp_pass', ''),
('smtp_from', 'noreply@procrm.com'),
('twilio_sid', ''),
('twilio_token', ''),
('twilio_whatsapp', '')
ON DUPLICATE KEY UPDATE id=id;
