<?php
require_once __DIR__ . '/../includes/functions.php';

if (isLoggedIn()) { header('Location: /client/dashboard.php'); exit; }

$error = '';
$success = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $db = getDB();
    $name     = sanitize($_POST['name'] ?? '');
    $phone    = sanitize($_POST['phone'] ?? '');
    $email    = filter_var($_POST['email'] ?? '', FILTER_VALIDATE_EMAIL);
    $whatsapp = sanitize($_POST['whatsapp'] ?? '');
    $telegram = sanitize($_POST['telegram'] ?? '');
    $password = $_POST['password'] ?? '';
    $confirm  = $_POST['confirm_password'] ?? '';

    if (!$name || !$phone || !$email || !$password) {
        $error = 'Please fill all required fields.';
    } elseif (!$email) {
        $error = 'Invalid email address.';
    } elseif (strlen($password) < 8) {
        $error = 'Password must be at least 8 characters.';
    } elseif ($password !== $confirm) {
        $error = 'Passwords do not match.';
    } else {
        // Check duplicate
        $stmt = $db->prepare("SELECT id FROM users WHERE email=?");
        $stmt->bind_param('s', $email);
        $stmt->execute();
        if ($stmt->get_result()->num_rows > 0) {
            $error = 'Email already registered. Please login.';
        } else {
            $hashed = password_hash($password, PASSWORD_DEFAULT);
            $stmt = $db->prepare("INSERT INTO users (name,phone,email,whatsapp,telegram_id,password) VALUES (?,?,?,?,?,?)");
            $stmt->bind_param('ssssss', $name, $phone, $email, $whatsapp, $telegram, $hashed);
            if ($stmt->execute()) {
                // Send welcome email
                $content = "<h2>Welcome to ProCRM!</h2>
                    <p>Hi <strong>$name</strong>, your account has been created successfully.</p>
                    <p>Your account is under review. You'll receive an email once approved by our team.</p>
                    <p>Once approved, you can login at: <a href='http://{$_SERVER['HTTP_HOST']}/client/login.php' style='color:#fff'>Login Here</a></p>";
                sendEmail($email, 'Welcome to ProCRM - Account Under Review', emailTemplate('Welcome!', $content));
                $success = 'Registration successful! Your account is pending admin approval.';
            } else {
                $error = 'Registration failed. Please try again.';
            }
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<title>Register — ProCRM</title>
<link rel="stylesheet" href="/assets/css/style.css">
<style>
.auth-steps { display: flex; gap: 8px; margin-bottom: 28px; }
.step { height: 3px; flex: 1; background: var(--border-light); border-radius: 2px; }
.step.done { background: var(--white); }
</style>
</head>
<body>
<div class="auth-wrapper">
  <div class="auth-brand">
    <div class="auth-brand-logo">ProCRM</div>
    <div class="auth-brand-tagline">Client Management Platform</div>
    <div class="auth-brand-desc">Your professional hub for campaign management and expert training services.</div>
  </div>
  <div class="auth-form-area">
    <div class="auth-box">
      <div class="auth-header">
        <h2>Create Account</h2>
        <p>Fill in your details to get started</p>
      </div>

      <?php if ($error): ?>
        <div class="flash flash-error"><span class="flash-icon">✕</span> <?= htmlspecialchars($error) ?></div>
      <?php endif; ?>
      <?php if ($success): ?>
        <div class="flash flash-success"><span class="flash-icon">✓</span> <?= htmlspecialchars($success) ?>
          <br><a href="/client/login.php" style="color:inherit;text-decoration:underline;font-size:12px;display:block;margin-top:6px">Go to Login →</a>
        </div>
      <?php else: ?>

      <form method="POST" novalidate>
        <div class="form-row">
          <div class="form-group">
            <label class="form-label">Full Name *</label>
            <input type="text" name="name" class="form-control" placeholder="John Doe" value="<?= htmlspecialchars($_POST['name'] ?? '') ?>" required>
          </div>
          <div class="form-group">
            <label class="form-label">Phone Number *</label>
            <input type="tel" name="phone" class="form-control" placeholder="+1 234 567 8900" value="<?= htmlspecialchars($_POST['phone'] ?? '') ?>" required>
          </div>
        </div>

        <div class="form-group">
          <label class="form-label">Email Address *</label>
          <input type="email" name="email" class="form-control" placeholder="you@example.com" value="<?= htmlspecialchars($_POST['email'] ?? '') ?>" required>
        </div>

        <div class="form-row">
          <div class="form-group">
            <label class="form-label">WhatsApp Number</label>
            <input type="tel" name="whatsapp" class="form-control" placeholder="+1 234 567 8900" value="<?= htmlspecialchars($_POST['whatsapp'] ?? '') ?>">
          </div>
          <div class="form-group">
            <label class="form-label">Telegram ID</label>
            <input type="text" name="telegram" class="form-control" placeholder="@username" value="<?= htmlspecialchars($_POST['telegram'] ?? '') ?>">
          </div>
        </div>

        <div class="form-row">
          <div class="form-group">
            <label class="form-label">Password *</label>
            <div style="position:relative;display:flex;align-items:center;">
              <input type="password" name="password" class="form-control" placeholder="Min. 8 characters" style="padding-right:48px" required>
              <button type="button" class="password-toggle" style="position:absolute;right:12px;background:none;border:none;color:var(--text-muted);font-size:16px;cursor:pointer;">👁</button>
            </div>
          </div>
          <div class="form-group">
            <label class="form-label">Confirm Password *</label>
            <input type="password" name="confirm_password" class="form-control" placeholder="Repeat password" required>
          </div>
        </div>

        <button type="submit" class="btn btn-primary btn-full btn-lg" style="margin-top:8px">
          Create Account
        </button>

        <p class="text-center text-muted mt-16" style="font-size:14px">
          Already have an account? <a href="/client/login.php" style="color:var(--white)">Sign in</a>
        </p>
      </form>
      <?php endif; ?>
    </div>
  </div>
</div>
<script src="/assets/js/app.js"></script>
</body>
</html>
